import React, { useEffect, useState } from "react";
import { FlatList, Pressable, Text, View } from "react-native";
import { useDispatch, useSelector } from "react-redux";
import NoContentIcon from "../../../assets/svgs/empty content/noAddress.svg";
import useAuth from "../../../hooks/useAuth";
import {
    fetchAddressStart,
    getMoreAddress,
    getMyAddress,
    getMyAddressWithoutLoading,
} from "../../../redux/slices/user/address/getMyAddress";
import { refresh } from "../../../redux/slices/user/util/refresh";
import AddressSkeleton from "../../../src/skeletons/screens/profile/AddressSkeleton";
import BackNavigation from "../../Utilities/CustomHeader/BackNavigation";
import EmptyContent from "../../Utilities/EmptyContent/EmptyContent";
import { ProfileStyles } from "../ProfileStyle";
import AddressStyle from "./AddressStyle";
import SingleAddress from "./SingleAddress";
import { useIsFocused } from "@react-navigation/native";
import { resetCountry } from "../../../redux/slices/user/address/getCountries";
import { resetState } from "../../../redux/slices/user/address/getStates";
import { resetCity } from "../../../redux/slices/user/address/getCity";
import { BASE_API_URL_STG } from "@env";
import useLangTranslation from "../../../hooks/useLangTranslation";
import CustomActiveIndicator from "../../../src/components/CustomLoader/CustomActiveIndicator";

const Address = (props) => {
    const { trans } = useLangTranslation();
    const dispatch = useDispatch();
    const { myAddresses, loading, isRefresh, nextPageUrl, loadMore } = useSelector(
        (state) => state.getMyAddress
    );

    const [inFileLoading, setInFileLoading] = useState(true);
    const { access_token } = useAuth();
    const URL = `${BASE_API_URL_STG}/user/addresses`;

    useEffect(() => {
        (async () => {
            let isMounted = true;
            if (isMounted) {
                await dispatch(getMyAddress(access_token));
                access_token && setInFileLoading(false);
            }
            return () => {
                isMounted = false;
            };
        })()
    }, []);

    const isFocused = useIsFocused();
    useEffect(() => {
        if (isFocused) {
            dispatch(resetCountry());
            dispatch(resetState());
            dispatch(resetCity());
        }
    }, [isFocused]);

    const onRefresh = () => {
        refresh(
            access_token,
            URL,
            dispatch,
            fetchAddressStart,
            getMyAddressWithoutLoading
        );
    };

    const handleLoadMoreAddress = () => {
        let isMounted = true;
        if (isMounted && nextPageUrl && !loadMore) {
            dispatch(getMoreAddress({ access_token, URL: nextPageUrl }));
        }
        return () => {
            isMounted = false;
        };
    }

    const [fastLoad, setFastLoad] = useState(true);
    useEffect(() => setFastLoad(false), []);
    if (fastLoad) return <View />;

    return (
        <>
            <BackNavigation
                navigationProps={props.navigation}
                routeName={trans("address book")}
                capitalize={false}
            />
            <View style={ProfileStyles.hrLine} />
            <Pressable
                style={AddressStyle.btnContainer}
                onPress={() =>
                    props.navigation.navigate("add new address", {
                        isUpdate: false,
                    })
                }
            >
                <Text style={AddressStyle.btnText}>
                    + {trans("Add New Address")}
                </Text>
            </Pressable>
            {inFileLoading || loading ? (
                <AddressSkeleton />
            ) : (
                <View style={AddressStyle.container}>
                    {myAddresses?.length > 0 ? (
                        <>
                        <FlatList
                            data={myAddresses}
                            renderItem={({ item, index }) => (
                                <SingleAddress
                                    item={item}
                                    index={index}
                                    navigation={props.navigation}
                                />
                            )}
                            keyExtractor={(item) => item.id}
                            refreshing={isRefresh}
                            onRefresh={onRefresh}
                            showsVerticalScrollIndicator={false}
                            initialNumToRender={10}
                            windowSize={10}
                            onEndReachedThreshold={0.5}
                            onEndReached={handleLoadMoreAddress}
                        />
                        {loadMore && <CustomActiveIndicator style={AddressStyle.loadMore} />}
                        </>
                    ) : (
                        <FlatList
                            refreshing={isRefresh}
                            onRefresh={onRefresh}
                            showsVerticalScrollIndicator={false}
                            ListHeaderComponent={() => (
                                <EmptyContent
                                    Icon={NoContentIcon}
                                    text={trans("No Address Has Been Fixed")}
                                />
                            )}
                        />
                    )}
                </View>
            )}
        </>
    );
};

export default Address;
